/*******************************************************************
 *
 *	File:		EMameEngine.cpp
 *
 *	Author:		Peter van Sebille (peter@yipton.demon.co.uk)
 *
 *	(c) Copyright 2001, Peter van Sebille
 *	All Rights Reserved
 *
 *******************************************************************/

#include "EikEnv.h"

#include "EMameEngine.h"

const TInt KGameStackSize	= 0x10000;			// 64KB
const TInt KGameHeapSizeMin	= 0x10000;			// 64KB
const TInt KGameHeapSizeMax	= 0x2000000;		// 32 MB




CMameEngine* CMameEngine::NewL()
{
	CMameEngine* self = new CMameEngine;
	CleanupStack::PushL(self);
	self->ConstructL();
	CleanupStack::Pop();
	return self;
}

CMameEngine::CMameEngine()
{
}

CMameEngine::~CMameEngine()
{
	if (iGlobalData)
		iGlobalData->iShutdownFunc();

	if (iGlobalDataChunk.Handle())
		iGlobalDataChunk.Close();

#ifdef __WINS__
	if (iMameLib.Handle())
		iMameLib.Close();
#endif
}


void CMameEngine::ConstructL()
{
#ifdef __WINS__
	iMameLib.Load(_L("emame.dll"));
	TLibraryFunction func = iMameLib.Lookup(1);
	(*func)();
#endif

	User::LeaveIfError(iGlobalDataChunk.OpenGlobal(KGlobalDataChunkName, EFalse));
	iGlobalData = *((TGlobalData**) iGlobalDataChunk.Base());
}



void CMameEngine::ShutdownGame()
{
#ifdef __WINS__
		/*
		 * Wait 1 second before killing it off. Thread's dying whilst running under the  
		 * may result in a emulator + devstudio freeze.
		 */
	User::After(1000 * 500);
	KillRunningStdoutConsole();
	User::After(1000 * 500);
#else
	iMameThread.Terminate(0);
#endif
	iMameThread.Close();
}


void CMameEngine::KillRunningStdoutConsole()
{
	TInt	id = 0;
	TBool	sentEvent = EFalse;
	RWsSession&		wsSession = CEikonEnv::Static()->WsSession();

		/*
		 * When EPOC's STDLIB shuts down it will put up a "press a key to close console".
		 * In WINS, we'll find the window group belonging to the thread STDOUT and send
		 * it a key event to have it automatically closed.
		 */
	while (id >= 0)
	{
		id = wsSession.FindWindowGroupIdentifier(id , _L("*"));
		if (id > 0)
		{
			TThreadId threadId;
			if (wsSession.GetWindowGroupClientThreadId(id , threadId) == KErrNone)
			{
				RThread	thread;
				if (thread.Open(threadId) == KErrNone)
				{
					TFullName	name;
					name = thread.Name();
					if (name.Compare(_L("STDOUT")) == 0)
					{
						TWsEvent	event;
						TKeyEvent	key;
						
						key.iCode=EKeyEscape;
						key.iScanCode=0;
						key.iModifiers=0;

						event.SetType(EEventKey);
						*event.Key()=key;
						event.SetTimeNow();
						wsSession.SendEventToWindowGroup(id, event);

						sentEvent = ETrue;
					}
					thread.Close();
//					RDebug::Print(_L("CMameAppUi: thread=%S, id=%d"), &name, id);
				}
			}
		}
	}
	if (!sentEvent)
		User::InfoPrint(_L("Please close the task named STDOUT"));
}


void CMameEngine::RunGameL(TInt aGame)
{
	TBuf<32>	threadName;

		/*
		 * Hmm, there is something funny about the game thread on target. Eventhough the
		 * previous game thread has finished (we even called Terminate on it) we cannot
		 * create a new instance of it with the same name (returns with KErrAlreadyExists).
		 * Using unique names gets round that.
		 */
	iGame = aGame;
	threadName.Format(_L("Game%d"), iGameCounter++);
	User::LeaveIfError(iMameThread.Create(threadName, CMameEngine::ThreadMainEmame,  KGameStackSize, KGameHeapSizeMin, KGameHeapSizeMax, this, EOwnerThread));
	iMameThread.Resume();
}


void CMameEngine::DoRunGameL()
{
	(*iGlobalData->iRunGameFunc)(iGame, iGameOptions);
}


TInt CMameEngine::ThreadMainEmame(TAny* aData)
{  
	CTrapCleanup* cleanup=CTrapCleanup::New();

	CMameEngine*	self = (CMameEngine*) aData;
	TRAPD(err, self->DoRunGameL());

	delete cleanup;

	return(err);
}


void CMameEngine::SetWindowGroupId(TInt aWindowGroupId)
{
	iGlobalData->iAppWindowGroupId = aWindowGroupId;
}

struct GameDriver**	CMameEngine::GameDrivers()
{
	return iGlobalData->iGameDrivers;
}

const char* CMameEngine::CpuName(TInt aCpuType)
{
	return iGlobalData->iCpunumNameFunc(aCpuType);
}


TInt CMameEngine::GameNo(const TDesC& aGameName)
{
	TInt	gameNo = 0;
	struct GameDriver**	game = GameDrivers();
	while(*game)
	{
		TPtrC8	gameName8((TUint8*) (*game)->description);
		TBuf<128>	gameName;
		gameName.Copy(gameName8);
		if (aGameName.Compare(gameName) == 0)
			break;
		game++;
		gameNo++;
	}
	return gameNo;
}

TBool CMameEngine::IsGameCloneOf(struct GameDriver* aGame)
{
	return (aGame->clone_of && !(aGame->clone_of->flags & NOT_A_DRIVER));
}
